import files from "./files";

type Encoding = "unknown" | "utf8" | "efbbbf" | "fffe" | "feff";
export default new class {
    public filePath!: string;
    public encoding!: Encoding;
    public bytes!: Buffer;
    public read(filePath: string): void {
        this.bytes = files.readBytes(this.filePath = filePath);
        const L = this.bytes.length;
        if (L) {
            switch (this.bytes[0]) {
                case 0xef:
                    if (L > 2 && this.bytes[1] == 0xbb && this.bytes[2] == 0xbf) {
                        this.encoding = "efbbbf";//utf8 bom
                    } else {
                        this.encoding = "unknown";
                    }
                    break;
                case 0xff:
                    if (L > 1 && this.bytes[1] == 0xfe) {
                        this.encoding = "fffe";//utf16 little-endian
                    } else {
                        this.encoding = "unknown";
                    }
                    break;
                case 0xfe:
                    if (L > 1 && this.bytes[1] == 0xff) {
                        this.encoding = "feff";//utf16 big-endian
                    } else {
                        this.encoding = "unknown";
                    }
                    break;
                default:
                    this.encoding = "utf8";
                    let i: number = 0;
                    while (i < L) {
                        const c1 = this.bytes[i++]!;
                        //0xxxxxxx
                        if (c1 < 0x80) continue;
                        if (i < L) {
                            const c2 = this.bytes[i++]!;
                            if (c1 < 0xe0)//110xxxxx 10xxxxxx
                            {
                                if ((c1 & 0xe0) == 0xc0 && (c2 & 0xc0) == 0x80) continue;
                                this.encoding = "unknown";
                                break;
                            }
                            if (i < L) {
                                const c3 = this.bytes[i++]!;
                                if (c1 < 0xf0)//1110xxxx 10xxxxxx 10xxxxxx
                                {
                                    if ((c1 & 0xf0) == 0xe0 && (c2 & 0xc0) == 0x80 && (c3 & 0xc0) == 0x80) continue;
                                    this.encoding = "unknown";
                                    break;
                                }
                                if (i < L) {
                                    const c4 = this.bytes[i++]!;
                                    //11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
                                    if ((c1 & 0xf8) == 0xf0 && (c2 & 0xc0) == 0x80 && (c3 & 0xc0) == 0x80 && (c4 & 0xc0) == 0x80) continue;
                                    this.encoding = "unknown";
                                    break;
                                }
                            }
                        }
                        this.encoding = "unknown";
                        break;
                    }
                    break;
            }
        } else {
            this.encoding = "unknown";
        }
    }
    public write(encoding: Encoding = "utf8"): boolean {
        let str: string;
        switch (this.encoding) {
            case "utf8":
                str = this.bytes.toString("utf8");
                break;
            case "efbbbf":
                str = this.bytes.toString("utf8", 3, this.bytes.length);
                break;
            case "fffe":
                str = this.bytes.toString("utf16le", 2, this.bytes.length);
                break;
            case "feff":
                this.lebe(this.bytes);
                str = this.bytes.toString("utf16le", 2, this.bytes.length);
                break;
            default:
                str = iconv.decode(this.bytes, "gbk");
                break;
        }
        switch (this.encoding = encoding) {
            case "utf8":
                this.bytes = Buffer.from(str, "utf8");
                break;
            case "efbbbf":
                this.bytes = Buffer.concat([Buffer.from([0xef, 0xbb, 0xbf]), Buffer.from(str, "utf8")]);
                break;
            case "fffe":
                this.bytes = Buffer.concat([Buffer.from([0xff, 0xfe]), Buffer.from(str, "utf16le")]);
                break;
            case "feff":
                this.bytes = Buffer.concat([Buffer.from([0xfe, 0xff]), this.lebe(Buffer.from(str, "utf16le"))]);
                break;
            default:
                this.bytes = iconv.encode(str, "gbk");
                break;
        }
        return files.checkAndWrite(this.filePath, this.bytes);
    }
    private lebe(bytes: Buffer): Buffer {
        let i: number = bytes.length;
        if (i & 1) {
            console.error("utf16 big-endian L = " + i + " 不是偶数！");
        } else {
            while ((i -= 2) >= 0) {
                const tmp = bytes[i]!;
                bytes[i] = bytes[i + 1]!;
                bytes[i + 1] = tmp;
            }
        }
        return bytes;
    }
}