import Utils from './Utils';
import files from './files';
import tempFiles from './tempFiles';

export interface Result { out: string, err: string };
export default new class {
    private sb: Array<string> = [];
    public standBy(): void {
        this.sb.length = 0;
    }
    public cd(path: string): void {
        this.add('cd ' + this.normalizePath(path));
    }
    public echo(msg: string): void {
        this.add('echo ' + msg);
    }
    public pause(): void {
        this.add('pause');
    }

    public copyFile(src: string, dest: string): void {
        this.add('copy ' + this.normalizePath(src, true) + ' ' + this.normalizePath(dest, true));
    }
    public cutFile(src: string, dest: string): void {
        this.add('move ' + this.normalizePath(src, true) + ' ' + this.normalizePath(dest, true));
    }
    public mkdir(dirPath: string): void {
        dirPath = this.normalizePath(dirPath, true);
        this.add("if not exist " + dirPath + " (");
        this.add("md " + dirPath);
        this.add(")");
    }
    public rmdir(dirPath: string): void {
        this.add("rd " + this.normalizePath(dirPath, true) + " /s /q");
    }
    public clearDir(dirPath: string): void {
        this.add('for /d %%i in (' + this.normalizePath(dirPath + '*', true) + ') do (');
        this.add('	rd "%%i" /s /q');
        this.add(')');
        this.deleteFiles(dirPath, '*');
    }
    public clearOrMakeDir(dirPath: string): void {
        this.add("if exist " + this.normalizePath(dirPath, true) + " (");
        this.clearDir(dirPath);
        this.add(") else (");
        this.mkdir(dirPath);
        this.add(")");
    }
    public deleteFiles(dirPath: string, pattern: string): void {
        this.deleteFile(dirPath + pattern);
    }
    public deleteFile(filePath: string): void {
        this.add('del ' + this.normalizePath(filePath, true) + ' /s /f /q');
    }
    public add(line: string): void {
        this.sb.push(line.replace(/\s*[\n\r]\s*/g, "\r\n"));
    }
    public async run(info?: string): Promise<Result> {
        const cmdPath = tempFiles.getFilePath('.bat');
        files.writeStr(cmdPath, this.sb.join('\r\n'), 'gbk');
        let re: (result: Result) => void
        const promise = new Promise<Result>(_re => {
            re = _re;
        });
        info && Utils.showLoading?.(info);
        execFile('cmd.exe', ['/c', this.normalizePath(cmdPath)], { encoding: 'binary' }, (error, stdout, stderr) => {
            const result: Result = { out: '', err: '' };
            if (error) {
                const msg = files.gbkBytes2Str(Buffer.from(error.message, 'binary'));
                result.err += msg + '\n';
                files.writeStr(tempFiles.getFilePath('.error.txt'), msg);
            }
            if (stdout) {
                files.writeStr(tempFiles.getFilePath('.out.txt'), result.out = files.gbkBytes2Str(Buffer.from(stdout, 'binary')));
            }
            if (stderr) {
                const msg = files.gbkBytes2Str(Buffer.from(stderr, 'binary'));
                result.err += msg;
                files.writeStr(tempFiles.getFilePath('.err.txt'), msg);
            }
            info && Utils.hideLoading?.();
            re(result);
        });
        return promise;
    }
    public normalizePath(path: string, quote?: boolean): string {
        path = path.replace(/\/+/g, '\\');
        return quote ? '"' + path + '"' : path;
    }

    // public getBuildToolsPath(): string {
    //     const exePath = Store.editor.exePaths["1.7.4"];
    //     if (exePath) {
    //         return exePath.replace(/Tuanjie\.exe$/, "Data/PlaybackEngines/WebGLSupport/BuildTools/");
    //     }
    //     console.error("无 exePath");
    //     return undefined!;
    // }
    public brotliCompress(inputPath: string, outputPath: string): void {
        this.add(this.normalizePath(Utils.exesPath + "Brotli/win_x86_64/brotli.exe", true) + " --quality 11 --input " + this.normalizePath(inputPath, true) + " --output " + this.normalizePath(outputPath, true));
    }
    public brotliDecompress(inputPath: string, outputPath: string): void {
        this.add(this.normalizePath(Utils.exesPath + "Brotli/win_x86_64/brotli.exe", true) + " --decompress --input " + this.normalizePath(inputPath, true) + " --output " + this.normalizePath(outputPath, true));
    }
    private getWASMOptEXEPath(buildToolsPath: string): string {
        return this.normalizePath(buildToolsPath + "Emscripten/binaryen/bin/wasm-opt.exe", true);
    }
    public wasmOpt(buildToolsPath: string, inputPath: string, outputPath: string): void {
        this.add(this.getWASMOptEXEPath(buildToolsPath) + " -Oz --dce --vacuum " + this.normalizePath(inputPath, true) + " -o " + this.normalizePath(outputPath, true));
    }

    public zip(zipPath: string): void {
        this.add(this.normalizePath(Utils.exesPath + "7z/7z.exe", true) + ' a ' + this.normalizePath(zipPath, true) + ' *');
    }

    public img2dds(imgName: string, fmt: "DXT1" | "DXT5", extra?: string): void {
        this.add(this.normalizePath(Utils.exesPath + "texconv.exe", true) + " -f " + fmt + " -m 1 " + (extra ? extra + " " : "") + imgName);
    }
    public img2astc(imgPath: string, d: number, astcPath: string): void {
        this.add(this.normalizePath(Utils.exesPath + "astcenc/astcenc-avx2.exe", true) + ' -cl ' + this.normalizePath(imgPath, true) + ' ' + this.normalizePath(astcPath, true) + ' ' + d + 'x' + d + ' -verythorough');
    }
    public astc2png(astcPath: string, pngPath: string): void {
        this.add(this.normalizePath(Utils.exesPath + "astcenc/astcenc-avx2.exe", true) + ' -dl ' + this.normalizePath(astcPath, true) + ' ' + this.normalizePath(pngPath, true));
    }
}